#include "Particle.h"
#include <algorithm>


namespace BARY
{
	//initialization of the evaluation point
	void Particle::init_particle(Vec3 p, int num)
	{
		m_pos = p;
		bary.assign(num, 0);
	}
	

	//normalize, such that the sum is equal to 1
	void Particle::normalize_bary()
	{
		double sum = 0;
		for (int i = 0; i < bary.size(); i++)
			sum += bary[i];
		assert(fabs(sum) > ZERO_TOL);
		if (fabs(sum) < ZERO_TOL)
			std::cout << "sum is close to zero!" << std::endl;

		for (int i = 0; i < bary.size(); i++)
			bary[i] /= sum;	
	}

	//find the maximum/minmum value of coordinates for visualization
	void Particle::find_coord_range(double& max_coord, double& min_coord)
	{
		max_coord = -1e20;
		min_coord = 1e20;
		for (int i = 0; i < bary.size(); i++)
		{
			max_coord = max_coord>bary[i] ? max_coord : bary[i];
			min_coord = min_coord < bary[i] ? min_coord : bary[i];
		}
	}

	//compute the weighted points for constructing the power diagram
	void Particle::compute_wps(std::vector< std::pair<Rt::Weighted_point, int> > &wpoints, DDG::Mesh& domain, 
		bool replug, int mode, double metric_xx, double metric_yy, double metric_zz, double alpha, double beta)
	{
		int p_num = domain.nvs();
		wpoints.resize(p_num + 1);
		//store for domain points
		for (int i = 0; i < p_num; ++i)
		{
			Vec3 p_pos = domain.vertices[i].position;
			double w = compute_weight(p_pos, replug, mode, alpha, beta, metric_xx, metric_yy, metric_zz, i, domain);
			Rt::Weighted_point wp(to_point(p_pos), w);
			wpoints[i] = std::make_pair(wp, i);
		}
		//store the point itself
		Rt::Weighted_point wp(to_point(m_pos), 0);
		wpoints[p_num] = std::make_pair(wp, p_num);
	}

	//compute the power coordinates
	void Particle::compute_power_bary(DDG::Mesh& domain, int mode, bool replug, double alpha,
		double beta, double metric_xx, double metric_yy, double metric_zz, double& max_coord, double& min_coord, bool output)
	{

		m_dual_faces.clear();
		int p_num = domain.nvs();
		std::vector< std::pair<Rt::Weighted_point, int> > wpoints;
		compute_wps(wpoints, domain, replug, mode, metric_xx, metric_yy, metric_zz, alpha, beta);//store weight points

		//build rt_mesh
		Rt cgal_mesh;
		cgal_mesh.build(wpoints);

		Rt::Vertex_handle vi = cgal_mesh.get_vertex(p_num);
		assert(vi != NULL);
		if (vi == NULL)
			std::cout << "hidden vertices itself!" << std::endl;

		std::vector<Rt::Edge> edge_vec;
		cgal_mesh.incident_edges(vi, std::back_inserter(edge_vec));

		//for each primal edge, construct its dual face
		for (int i = 0; i < edge_vec.size(); i++)
		{
			Rt::Edge current_e = edge_vec[i];
			Rt::Vertex_handle vj = cgal_mesh.get_target(current_e);
			if (cgal_mesh.is_infinite(current_e))
				continue;
			std::vector<Vec3> dual_points;
			Rt::Cell_circulator circ = cgal_mesh.incident_cells(current_e);
			Rt::Cell_circulator end = circ;
			CGAL_For_all(circ, end)
			{
				dual_points.push_back(to_vec3(cgal_mesh.dual(circ)));
			}
				
			m_dual_faces.push_back(Dual_Face(dual_points, vj->info()));
		}
		
		
		store_bary_from_dual_edges(domain);

	}


	//compute the weight value for the given position
	double Particle::compute_weight(Vec3 p, bool replug, int mode, double alpha, double beta, 
		double m_xx, double m_yy, double m_zz, int v_id, DDG::Mesh& domain)
	{
		double l = (p - m_pos).norm();

		
		if (replug)
		{
			
			return (l*l - pow(l, bary[v_id]));
		}
			

		switch (mode)
		{
		case(DHC) :
			return 0;

		case(WAC) :
			return(l*l - 2);

		case(MVC) :
			return(l*l - 2 * l);

		case(HALF) :
			return(l*l - 2 * std::sqrt(l));

		case(QUAD) :
			return(l*l - 2 * pow(l, 4));

		case(ANISO) :
			return(l*l - length_aniso(p - m_pos, 5*pow(cos(m_xx*m_pos[0]),2)+1, m_yy, m_zz));
			break;
                default:
                    std::cout<<"default DHC mode!";
                    return 0;
		}

		return 0;
	}
	
	
	//compute the barycentric coordinates given the size ratio of the duals and primals
	void Particle::store_bary_from_dual_edges(DDG::Mesh& domain)
	{
		int p_num = domain.nvs();
		bary.assign(p_num, 0); //clear
	
		for (int i = 0; i < m_dual_faces.size(); i++)
		{
			int v_id = m_dual_faces[i].v_id;
			Vec3 primal_e = m_pos - domain.vertices[v_id].position;
			bary[v_id] = compute_polygon_area(m_dual_faces[i].dual_points, primal_e) / primal_e.norm();
		}

		//normalize, such that the sum is equal to 1
		normalize_bary();
	}
	

}
